package main

import (
	"fmt"
	"io/ioutil"
	"os"
	"os/exec"
	"os/user"
	"path/filepath"
	"regexp"
	"strings"
	"time"
)

const (
	Ver     = "2.0"
	LastUpd = "Jan. 14, 2026"
	Author  = "Chenf"
)

var (
	MyFileList   []string
	MyIgnoreList []string
	NotGotoTrash = 0
	HelpFlag     = 0
)

func main() {
	// get hostname
	trashroot := "/home"

	// get current login user name
	currentUser, err := user.Current()
	if err != nil {
		fmt.Println("Error getting current user:", err)
		os.Exit(1)
	}
	username := currentUser.Username

	// mkdir trash dir if not exists
	trashpath := fmt.Sprintf("%s/%s/.Trash", trashroot, username)
	if !pathExists(trashpath) {
		os.MkdirAll(trashpath, 0755)
	}

	dateTimeDir := time.Now().Format("20060102150405")

	// handle options
	args := os.Args[1:]
	for _, arg := range args {
		switch {
		case arg == "-h" || arg == "-help" || arg == "--help":
			printHelp()
			os.Exit(0)

		case arg == "--version":
			fmt.Println("rm2mv v" + Ver)
			os.Exit(0)

		case arg == "--force":
			runCommand("/bin/rm", "-rf", arg)

		case strings.Contains(arg, "/.Trash/"):
			runCommand("chmod", "777", "-R", arg)
			runCommand("/bin/rm", "-rf", arg)

		case arg == "--status":
			if !pathExists(trashpath) {
				os.Exit(0)
			}
			runCommand("du", "-sh", trashpath)
			os.Exit(0)

		case arg == "--clean":
			cleanTrash(trashpath, dateTimeDir)
			os.Exit(0)

		case !strings.HasPrefix(arg, "-"):
			// protect /xxx and /home/xxx
			if isDangerousPath(arg) {
				fmt.Println("[Warning] deleting dangerous files: " + arg)
				os.Exit(0)
			} else {
				if pathExists(arg) {
					MyFileList = append(MyFileList, arg)
				} else if isSymlink(arg) { // invalid symlink
					MyFileList = append(MyFileList, arg)
				} else { // ignore non-exists file
					MyIgnoreList = append(MyIgnoreList, arg)
				}
			}

		default:
			MyIgnoreList = append(MyIgnoreList, arg)
		}
	}

	// check and clean trash
	trashtimestamp := trashpath + "/clean.timestamp"
	if !pathExists(trashtimestamp) {
		runCommand("rm", "--clean")
	} else {
		content, err := ioutil.ReadFile(trashtimestamp)
		if err == nil && len(content) >= 8 {
			lastDatetime := string(content)
			if len(lastDatetime) >= 8 {
				y := lastDatetime[0:4]
				m := lastDatetime[4:6]
				d := lastDatetime[6:8]
				lastTime, err := time.Parse("20060102", y+m+d)
				if err == nil {
					if time.Since(lastTime).Hours() > 7*24 {
						runCommand("rm", "--clean")
					}
				}
			}
		}
	}

	// remove files
	if len(MyFileList) != 0 {
		target := fmt.Sprintf("%s/%s", trashpath, dateTimeDir)
		if !pathExists(target) {
			os.MkdirAll(target, 0755)
		}

		for _, f := range MyFileList {
			targetFile := target + "/" + filepath.Base(f)

			// avoid deleting one file twice
			if pathExists(targetFile) {
				runCommand("/bin/rm", "-rf", targetFile)
				moveFile(f, target)
			} else if isPipeFile(f) {
				// handle special file type (named pipe)
				os.Remove(f)
			} else if isSymlink(f) && !pathExists(f) {
				// delete invalid symlink
				runCommand("/bin/rm", "-rf", f)
			} else if shouldDeleteDirectly(f) {
				// delete file directly by blacklist
				runCommand("/bin/rm", "-rf", f)
			} else {
				// move to ~/.Trash
				if pathExists(targetFile) {
					runCommand("/bin/rm", "-rf", targetFile)
				}
				moveFile(f, target)
			}
		}
	}
}

func printHelp() {
	fmt.Println("rm2mv v" + Ver)
	fmt.Println("a script to replace system rm, moves dirs or files to ~/.Trash instead of deleting them directly.")
	fmt.Println("e.g. rm2mv xx.v            delete a file")
	fmt.Println("     rm2mv -rf rtl         delete a directory")
	fmt.Println("     rm2mv --force simv*   delete file or dir not going to ~/.Trash")
	fmt.Println("     rm2mv --clean         clean files 1 week before from ~/.Trash")
	fmt.Println("     rm2mv --status        calculate the size of ~/.Trash")
}

func pathExists(path string) bool {
	_, err := os.Stat(path)
	return err == nil
}

func isSymlink(path string) bool {
	info, err := os.Lstat(path)
	if err != nil {
		return false
	}
	return info.Mode()&os.ModeSymlink != 0
}

func isPipeFile(path string) bool {
	info, err := os.Stat(path)
	if err != nil {
		return false
	}
	return info.Mode()&os.ModeNamedPipe != 0
}

func isDangerousPath(arg string) bool {
	if arg == "/" || arg == "~" {
		return true
	}
	homePattern := regexp.MustCompile(`^/home/\w+/?$`)
	rootPattern := regexp.MustCompile(`^/\w+/?$`)
	return homePattern.MatchString(arg) || rootPattern.MatchString(arg)
}

func shouldDeleteDirectly(f string) bool {
	// check file extensions
	extensions := []string{".fsdb", ".vcd", ".sim.history", ".bmp", ".dat", ".log", ".ucli.key", ".xrun.key"}
	for _, ext := range extensions {
		if strings.HasSuffix(f, ext) {
			return true
		}
	}

	// check if filename is "simv"
	if strings.HasSuffix(f, "simv") || filepath.Base(f) == "simv" {
		return true
	}

	// check patterns
	patterns := []string{
		`verdiLog`, `novas`, `xcelium\.d`, `waves\.shm`,
		`csrc`, `simv\.daidar`, `cov_work`, `INCA_libs`, `\.lib\+\+`,
	}
	for _, pattern := range patterns {
		matched, _ := regexp.MatchString(pattern, f)
		if matched {
			return true
		}
	}

	return false
}

func cleanTrash(trashpath, dateTimeDir string) {
	if !pathExists(trashpath) {
		return
	}

	files, err := ioutil.ReadDir(trashpath)
	if err != nil {
		return
	}

	if len(files) > 0 {
		fmt.Println("Clean Trash before 1 week...")
	}

	dtPattern := regexp.MustCompile(`^20\d{12}$`)

	for _, file := range files {
		dt := file.Name()
		if dt == "clean.timestamp" {
			continue
		}
		if !dtPattern.MatchString(dt) {
			continue
		}

		// parse date from directory name
		year := dt[0:4]
		month := dt[4:6]
		day := dt[6:8]
		dirTime, err := time.Parse("20060102", year+month+day)
		if err != nil {
			continue
		}

		if time.Since(dirTime).Hours() > 7*24 {
			dirPath := fmt.Sprintf("%s/%s", trashpath, dt)
			fmt.Printf("remove %s\n", dirPath)
			os.RemoveAll(dirPath)
		}
	}

	// write timestamp
	trashtimestamp := trashpath + "/clean.timestamp"
	ioutil.WriteFile(trashtimestamp, []byte(dateTimeDir), 0644)
}

func runCommand(name string, args ...string) error {
	cmd := exec.Command(name, args...)
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr
	return cmd.Run()
}

func moveFile(src, dstDir string) error {
	// replace rm with mv for safety
	cmd := exec.Command("mv", src, dstDir+"/")
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr
	return cmd.Run()
}
