## Subresource Integrity

If you are loading Highlight.js via CDN you may wish to use [Subresource Integrity](https://developer.mozilla.org/en-US/docs/Web/Security/Subresource_Integrity) to guarantee that you are using a legimitate build of the library.

To do this you simply need to add the `integrity` attribute for each JavaScript file you download via CDN. These digests are used by the browser to confirm the files downloaded have not been modified.

```html
<script
  src="//cdnjs.cloudflare.com/ajax/libs/highlight.js/11.4.0/highlight.min.js"
  integrity="sha384-f0iFMt2GY8IUGtC3kMLjHO6pxu4yJHCyCoPSHC3KskKtqVVKO8z8RdDDqIsUMm/Y"></script>
<!-- including any other grammars you might need to load -->
<script
  src="//cdnjs.cloudflare.com/ajax/libs/highlight.js/11.4.0/languages/go.min.js"
  integrity="sha384-2E3CF353IHVQzXELyO/v6Bujc+mALGvLJ/uYz4Wfga/JUnFpGjdzs0bvafwAIAhB"></script>
```

The full list of digests for every file can be found below.

### Digests

```
sha384-f0iFMt2GY8IUGtC3kMLjHO6pxu4yJHCyCoPSHC3KskKtqVVKO8z8RdDDqIsUMm/Y highlight.min.js
sha384-cONdclhP5KugJbiKf1VYq+c6HsIxmyQWbadFgI6DMT8YhCJdpIxPcxrJweqSR4FW highlight.js
sha384-vrzKp51bTFHBkJWAqEAT9fRwlZi0T3z54IFph8ynlke02LIDIsO8Oh/NwtliF25A es/highlight.min.js
sha384-yI0VdoaRPgwBgUN62dkTCAW7vIgrzB4huAfiViIs8/JBCDr0oGdR5WFwml5PfUQe es/highlight.js
sha384-vrzKp51bTFHBkJWAqEAT9fRwlZi0T3z54IFph8ynlke02LIDIsO8Oh/NwtliF25A es/core.min.js
sha384-yI0VdoaRPgwBgUN62dkTCAW7vIgrzB4huAfiViIs8/JBCDr0oGdR5WFwml5PfUQe es/core.js
sha384-ojptpLlxM2Okn7yphBroLvuL05KWy57S10WrKB+UEYKUcXsRWhZC/Xm7aQOuMjVH languages/1c.min.js
sha384-SYA83F16D8/A9J+HqelAX6v+OZpFMJKPkWzjSCLC+ohNmbTzhwg9JFeAag5zi5N+ languages/abnf.min.js
sha384-ADqc3U53blJ/rQafu/hMH5V89BmxQnIC8b+frsBXOui8OQHtsKmDmtRxoub1kgLS languages/accesslog.min.js
sha384-wdP21PJE2Zs0NAKuxG76e9evGmQsRe4oO9PTVGj/vlfNSAuiiikk7CAqs7JOq3+g languages/ada.min.js
sha384-0ECdJPZdJmawImtRrm1wxSS3RIz5eILnE7qHZ7AsfJfWbJNovJBB4va3a6jowkXQ languages/actionscript.min.js
sha384-GcZizNA8BZvKaxM2wNVaGhbQidbeZl2JVvQpBXQS30d98m6gxSmrz13eUq8OHYUZ languages/angelscript.min.js
sha384-Dm3m4Kyl92dhwa3K32tAPCWnRRgY4faDEDFsB6OxoynM3DWB9potrIwNEcyF+GWb languages/apache.min.js
sha384-Pp7IqjZLZeyD0yToSLP3hYZYMS1kn0U2X1YbS3kCVSG7X5cBIzg5X55Odb73cfbP languages/applescript.min.js
sha384-ZtdYIfG7h7GYQTy3D/cJ1BIMdvPfgfkrnceT2e7ZG02u4qFzIwvczu1ooqaecCQi languages/arcade.min.js
sha384-/jNwBM9uUrvRwIkBjd8YDyQSb+8z/2Vb5K+HHH9A6Co4aqIK0orYWExBiWen5XUv languages/armasm.min.js
sha384-HvHvcVBsGw2wJGzyp1YiRXS2j37PJFENDyzvkFr7ZzG+zKsww3hIVE97s6XClrNA languages/asciidoc.min.js
sha384-yIJOWNOiqa+TJt6n0l1GQ7gQxa7ghDjUO6KcGbMzx+FChmhzzjpEmmbdrPwRbE6q languages/aspectj.min.js
sha384-0R7+M/3GLVV8eVLQj+JQwuRImX5UOhnQWilpuEFxBf817OCIN7gB6/iiDZy51nWh languages/autohotkey.min.js
sha384-zvfssDlIKr8taEWaJBBn6jzB0YWpJRGMs7hF2LfjhrpA3qESScuUS5ZsnkqESbDs languages/autoit.min.js
sha384-YNzP9u/9FuE4L8yxDK/VQPhdcWFILK0QOKcMMeQSSFz0meY3zQ48fX9sBzACSzal languages/avrasm.min.js
sha384-UOyFnLIcF8Ikx7W2L78Mj5HlMe+WYU/6VvIl6o/NHnen0kFtkc3iQo4cwTLLZe2n languages/awk.min.js
sha384-SFYZZH2eUgcd9FTPQiqWNPhM3Ea3ZhZtLLbnq4+xuro2hihqGI9TAlGH+Vd9xFFr languages/axapta.min.js
sha384-byn6QFeCzjxYAj+7OgRKBITKR0JCfL5EXSVFY3Q7rmiF9TnAVYs4rfE+nofMWvfU languages/bash.min.js
sha384-PSQzTedk5I3ZZcxVsrYy5EUQd9QEkAFfBFvlrHQUIrMQ/1R42lLSBZBkaiqMnAXW languages/basic.min.js
sha384-EE0OkvjLP4+HGaV9G8beyUKhCm62ys8uoG0MXJOcOOM7mYR6DG+G8H4MeaWGwg1S languages/bnf.min.js
sha384-/lhjgdL4WtRokYATU7LO3zYLLuNPVDs0p1Ihg9TanQ4W714vMF+/G5F4QbM9fMzq languages/brainfuck.min.js
sha384-otKlFuNJtYrggKpTgTSKUF+3YM4eQYcLZ6Nu51z5Uy1izmU1dAK47vPc5Hd8lrO9 languages/c.min.js
sha384-MxmtwXLi/LSNgQvr+KhcXtHlZJkx5IgvDHQg22Tj806t6kt2guJa7AiSl1L7TFJe languages/cal.min.js
sha384-MlddQSiFlvDjZdrPjX8AsDah9GgbkDSzUc+l22iOfEbJuq73uJ7COYyIFXlzI84x languages/capnproto.min.js
sha384-yvtgT4HroqETN0bZU5sSo2kuO5lYDLnBy1voiRnJNbFMNF6J53tjD+zoxZH1uDzA languages/ceylon.min.js
sha384-6ETQqm+hlTlikHbeaoUXRq4io2fgItCQYizcGhLEoIomX/pxko1w31xfZ8hrU0Cb languages/clean.min.js
sha384-BtN6edE41m7IGEy0jcBvjPLU0XFmv0IozbyEtIlw4JMQcLcfN+TxrHLbrM5LkpTO languages/clojure-repl.min.js
sha384-tK/PYsxTx1+dfubsV0w4+V9OPGilwX8/y0I11nWGoEaV/S/vb3Ir6JvVxzDtS5LA languages/clojure.min.js
sha384-ZzPtJG2RrgIn9j5G7+A3ZLpfNBLHmQbWnJMMwj7PMFZputxYCF74g4vyTXphPlCt languages/cmake.min.js
sha384-ER8K00W/w3liLSAoJioxzSutbiZAjVz635bRrE4NV2Wuw7SYX1UJdnOs8wkYpdWG languages/coq.min.js
sha384-LXg81ejWqeFkUbiy92e81ryzzNeALB48CsmrC1v+cb7kwxYxP1KkPoc6UcX/MAx5 languages/cos.min.js
sha384-t3DdFNFQrpIyYimCsb02kDNcK9wh4vQdQFHj5As1QcSkg8AczIkLIjWwMd13uuOA languages/cpp.min.js
sha384-z1nQTUYqdxQZmVROWSd+JIq4dUH2rTGkUwlrUCetKcJ21or1Z/myq0mLhdVwwOhx languages/crmsh.min.js
sha384-W/nO1xzS3FohuWBu/EJdXP6uYkCi69rr2chruyz9sUlRjDchMUmgeqHjjL3MVBU7 languages/crystal.min.js
sha384-SxONKgNynL7yA2Ehyy496w9Yc+yjwpsLONFiYHyZSFzfZvPT4VFLAGcQqyTPxS5m languages/csharp.min.js
sha384-Ik20btD+IPcxTufkXchdk08XsNcKnLt3Pdg2gZczFoRDNUZ2DpescolkI9OBYxuV languages/csp.min.js
sha384-6NAgELfHU4f3GjDpUnYM5cckz4ojqHzPqp/G2/1rOEMJMHru+V5MdkZzV8+RrT8r languages/d.min.js
sha384-JCH/q+34vIHUkXUM5PH2Ao5uLMI6GD/igllbF8gryJ6mtBsiAALnyqxFTltcHvn5 languages/dart.min.js
sha384-zzGwk/je8FefmAXNOkm3+kjAN4VcDsrK7Z8vtdQ2P2YFW8QT8+L78l3Taovt69MW languages/delphi.min.js
sha384-24YD969Z0G4fX9+zIod9zi/cDJBBjc9gkZ7JvrRgeay259kYVIviFB9HVh4ykSpC languages/diff.min.js
sha384-5TqEh8DhOJaDi7qZx3Yi8ziFBCyxTBFI98xTih2IOF2oU8JmNQdTCYJMW9u/iTiu languages/django.min.js
sha384-LINUsfxzLwJTZ6jAdHbcdW2JF95IynsT0eoqIeXVAqVjeqMv9SDgbz3pjmuuaoz3 languages/dns.min.js
sha384-W28TUK27SJhMnSKG9JUi/7HmOPKXcRRaLBOfOVbFpgpfSegYF2S9iuQ1qMam6QC0 languages/dockerfile.min.js
sha384-kwLnLszJilnJz0KQgJPzBjkkOiyg8dmtRVyzISsmjIJB6nl7j6i5tT4VMJ+28YmR languages/dos.min.js
sha384-bhE9tXCOhNt41CYOtOqktMORMr5nmUeqPz7LIh1P0fYa9NbiUY8dGqk20ayDz2d2 languages/dsconfig.min.js
sha384-qzW3fCuMyLkTRqX6NsmZQQJQA3Dbq2rcXZSaoIvVCwNATN41x30GR71VXaBxRwcT languages/dts.min.js
sha384-Vc9aBXEdwBLGEDmnszWidUVoMTLRAPsWMygAXYDO+2qE4I8Z4SVcomx7SDLiiGir languages/dust.min.js
sha384-pwsVyD8ARgcOQJNY/6zKnik5oy9/9/8upIbzO8OecQ4932ACqOZj4A9M+ECtM6h6 languages/ebnf.min.js
sha384-JKmt6MhcYw6Yb7z1Z1v2ttUt+9M+EXkgpV8xnN8dTL6gbNX4XGl2heDW6GOL67lu languages/elixir.min.js
sha384-1a691zH1ORXanF4aOBcgfgCRhn4PVlMNQUBRtyB9Gu0cZtVhfcIaoOHabNAXv/qK languages/elm.min.js
sha384-3EKlwTG584xdvuoMwWhnxIyRvg06UptaMmUzC0IBF8NoFPVnt4/+hG1mEuGngQvR languages/erb.min.js
sha384-7/izO1AlbYhSlRwUiQN64ggSNUKgnrvlNwKMneFiU9akpqbFvKHwia1TeLUnboXR languages/erlang-repl.min.js
sha384-qik4LvRRgl+AAWrC56Ad9uPdFLE3xJ9X7O0jHgDoGW7Ljw9bc/1yfGKdSUndQMMq languages/erlang.min.js
sha384-CWXknh7VyKIRH1MYHs1RTAIAtS8CDU4pRzBwkURSaCHqD/Me08Lfq5w9mzrUwPxa languages/excel.min.js
sha384-G054tGdyiR33p7f0HeI6mfWF0RzHF05iJuSNQ9tquu1sBoBu0qWOSUJFadQtD5TG languages/fix.min.js
sha384-VIGPxMWIwB9R+8e/Cpl7FKQH5dLdFUv4GydnoG2WRiO6IpOZDqzngxPlaV2uJBbJ languages/flix.min.js
sha384-l5r83l5KcKjoHVDPMl8ewwQCSKVhrqcdkzzZSGcqUZC9ENGqhIUt6d5BD9wNZo0T languages/fortran.min.js
sha384-uyp8WLu/uhPCBNd0JUEj9wzVJDAY8bUc9raBboWc/n6RnM5JjHcO79i/nwPZgRlo languages/gams.min.js
sha384-YviesWEfLINbeFRq/qhqgNOlaDC0RW/drSmZclc0YubUClidUaeSGyJsVDGy+bQ5 languages/gauss.min.js
sha384-22ImtDhutWpFsLF5np0d+hxC6cVVB+SWoV0+Sb6IIA1IT1jHVnHbd0ZTnuyMgAYW languages/gherkin.min.js
sha384-QJo7FHWsZqKonsNtYJKqwr1c1N17y/hWto2bNwr65Xj+dFs5Uy+fqbs/T7BCvfcF languages/glsl.min.js
sha384-JGVMz+UneOuhqhKMImblk6UQgtOyy03f84iNTkeh804Z66qDgh7uCF0nER+qw7uw languages/gcode.min.js
sha384-miVc2uVtY6lFQg7S0ibmPaIHuL988AX59aBqgTE+ZYU2NiUr5cXwm390RZq298zb languages/gml.min.js
sha384-2E3CF353IHVQzXELyO/v6Bujc+mALGvLJ/uYz4Wfga/JUnFpGjdzs0bvafwAIAhB languages/go.min.js
sha384-OntJ+b4Rn9lN7mpacDIAGJXldNJJ5+Ky2poFsGT7PN6EuqxGhOUzawZWSK52XTR6 languages/gradle.min.js
sha384-B2ZfRgRWKwbicc1OH949c8jIIUUPifISyH65QvZvYOOg0nmNvSZnHPq/kGqhKzOk languages/golo.min.js
sha384-Bj4jZBJiJM9axKw5/Cf4y34Bth+J53vNHdRoVJilvJvVU88h307IBBi3dORZS/rt languages/haml.min.js
sha384-LQdScLKHpW/UboJ9mEvlCG5h1b4otB0yit/Pv8Vhx342khNLjyqTlw0O6wNkVAXZ languages/groovy.min.js
sha384-772osjcJyGGn7jePGnZ7nruIMso8w5HzLP4C5abWq7G7EpIPxQbjtBQknbQWeDdU languages/handlebars.min.js
sha384-bruYlGJ8iIb0GEMEbMndESM1ap+2Q2vTPR0xHsxF4yXZs6q861DjsNeZ//i1GCcb languages/haskell.min.js
sha384-PoTEw2vaY7FzLgifD6sKFNmB3WNM5S1DoV/1uGozF3cBCgI1atQjrDLgGN4somlq languages/haxe.min.js
sha384-lKcuwH5mw/xNHypV8gvtJJYW94RKkvbqdgV1W9U+Zk7j6b/hxj+kmjDv0lcLnUPt languages/hsp.min.js
sha384-jMSDfKLQYWcz0BTEIQ8lJbM98Q1MyTc03xTbAz64WKiQgj13iXCGMsbIfjWB+nMZ languages/http.min.js
sha384-ECbMZ1BFj3+i9ziofzhw+GPI5uXZwpq5yLeNuos/DYhqTzuZdP2CsYFfdHSGnvum languages/hy.min.js
sha384-I28t29+s6hB76hsZhbiyqb6ByNqXLmnV71HOk2/3VMCWFxsiUavbFzQpUHhwh08k languages/inform7.min.js
sha384-WhLOAQsZ6ZXQxngxUntRTI23s8jZF30n1ZRqoahKb560D7ueibtQxlaxeSCouY7i languages/ini.min.js
sha384-mG6FAKPO+cKF+M86bHLq3/0f9r0BmbhGb/B7lFziJj00Gp5TYwlRyMR17gpw+UNQ languages/irpf90.min.js
sha384-UrbUvD3f9J2ofzboq0Uj8cjKllFf9GFkVuRYOSwj4zYtDYpB8m1KL4kaKha3Y7u2 languages/isbl.min.js
sha384-8ZrYPPwSO8pp6WcJZp7hIavl9rSQthwaTCJi3b6wiLhd2aiZAy4yThs2+33cGxRg languages/jboss-cli.min.js
sha384-yJay20tE1VHRn0s51KD15MeWnhWrW4+RgSqdU04OahqNyaeabC7pJu6I6WveTHxN languages/json.min.js
sha384-Y55+F+r7dqYvRWCsgM+zbzVgSHgdA6QTePqSPpoG58i5Y6xsMxJz98NJjWjRZBhI languages/julia-repl.min.js
sha384-knu2/GjmwoG9fPm871WfoMm1IZbKZPvRITljAeW0bdWcTaO/1LQE5GluV4mFgCYH languages/julia.min.js
sha384-wjDxepwmAWzU53D4+4zjC7FBjmFpu4vuc+h0KAP0W3+7FPSz4v09sUddOYMyiY21 languages/lasso.min.js
sha384-GdrPRlLC8KQGj5LwkLBMF4fOwqpl7nqNRLyftjIde1yDEA9nvmJICoWhN+x2nn5z languages/latex.min.js
sha384-kXiIXUxs31rW7VHOCd9ttv/2vtKmolc9n67biNPR4ozfrtC7/AP+8OtXVoL1rPDu languages/ldif.min.js
sha384-1mus0kLy5O5b+5J8Y3ucA7Ot6tRhQn5Vbsf/2qf8vvkLvFz/+5i5LjEz159YvGXs languages/leaf.min.js
sha384-IjeujREzDs/2/7ntfjWBtpx33s+rcNqGc/Whv63YcKrnO7nXRg73BxQgsp3sg31O languages/lisp.min.js
sha384-4R1QInwKb6Ka2R5jYFkQDVBkLhM5YFewrjV6B7s/pSORQyLOKBCyNGVTkyExXh0V languages/livecodeserver.min.js
sha384-5zi/SziismipQDkoWzfRcj6A1wQf1mh1b3e7gqmLfF/UixIHBg9IMvNr/3wUuuFr languages/llvm.min.js
sha384-1frhuDpVwoySonRQreuGMKXrlQjAcghfiGEsS+wbU5M3m5VqD7dIn9R9Z3YXp6Af languages/lsl.min.js
sha384-Ma/K2qtjBCjrwW+R259IAQ8CkUKy6bfjyURhDvyFalPFWRlso1zxiqZONPRDj1bc languages/lua.min.js
sha384-Prrpd1PNr3Ftd+67oS/AawWvSkAREY1XVBM5Rf5HpFguEDDddqRELyIMCNVIhiI4 languages/makefile.min.js
sha384-MUldR+IqA9sWfDDKKI4Il0zNCuegL4dzwvlchUF+XaMW2IwLb2kMkHc03Tj22aYU languages/markdown.min.js
sha384-1Lkhezlb5FHH2j+/xyl5uX6GbEfnUUHeSOyUUT4eMzRppW8H701frxoZeAqkUFo9 languages/matlab.min.js
sha384-VltZOHM5I+7MAZdBNy6rPg5jxlnjbp2LTfvI5Pv/bKuirB8rBbQYSM6o8/Z11he1 languages/maxima.min.js
sha384-b/jTxPDD5qMEvoWskDD7R1xUCY2Pkv4jSIeubKYadi/cGbNKzofNat14Fo+8qdRg languages/mel.min.js
sha384-UQv8uYVXYcINWiwM+DpHen6rS+IA+mUigDh/ZDr52M9JDeYj7Yw3OIrINilIhFhq languages/mercury.min.js
sha384-DMQDQF2GZ0gtcWEfZAsl1N0yUL/LT2H1zvw2ah8bR91XniPvHT34LA6zsJ9V65A+ languages/mipsasm.min.js
sha384-74uvdFj1vnNepLOZUs/pWO8s8JHofMmuYdD5cIGZdkcpTnpDQpw5auEWgbnB6ick languages/mizar.min.js
sha384-mkH9YT6K1oIfWsQ04UvxnmBpED8QRf7SiZFtUP2FNquSFdHK+nXZ23QoOiCESvFS languages/monkey.min.js
sha384-q1tVnjqZbXAVQtUrpzl7OAZybtjzb6/c6Gre4W6LmsHRPkP8GBvbPbCTgkoKj6Ey languages/mojolicious.min.js
sha384-vC3IZFHMsh4Uj42XGNxTw+D3oZdG8QdKynUM8dg/lsLX2pquUmDrSXQX8ZfnVSB7 languages/moonscript.min.js
sha384-TSep8pxyI/IYft298d7Vpml/lMJRD49nAr3v5KRtjHvIWTourR/z8srUKTFokn9I languages/n1ql.min.js
sha384-cSdSMoQB4r/oPNejBPFeZf2RlQ/jutTwI7+C99mJm9HOGmyAvIi5irlyi8NpqpCv languages/nestedtext.min.js
sha384-K/LzmL2wddBkawY37ZVAO/rToMcut78XmqPEUuNA7kKIGEcEszldITNgHEcgkbIG languages/nginx.min.js
sha384-DFY1abJ1K6bFsRNGA6wnzTDBPOKrafb9mzzmWhyIz++7ewihLKhNVnmZmRJrK+yk languages/nim.min.js
sha384-Qr/4xOXqIau/g32XSWThmAUPVT3f4t7+PEMFuZM7bNxsfHzFSllsXpZ+55DS04lU languages/nix.min.js
sha384-S/p2rQToGyhZCMbSz4i3GiS08qzBQD65A6pwyxyPSpZDPmB04V3moxpfC8Qa30tr languages/node-repl.min.js
sha384-GGL9W1xA3G+OnJ5m/buXau+HVfH1RDPLzmwaGpm/FDmE8p9nCPZEZNcQBg58TkR4 languages/objectivec.min.js
sha384-wp6FvoNEZUECQdGwcMi3ouU0pOyDacR5atoWtLnmNiGekCN3lUEs03yvimO+3nZ4 languages/ocaml.min.js
sha384-eDT+b3zdng8nLImxsMr6A/06OH2B2VCVU4OwtqtV49Layhd5R4RZsoqXVYzuRTnz languages/openscad.min.js
sha384-MEG4QAlF+7hMSR3gI095G5U/cgZ2C7TW1Kw0jvDrcvoS7e+zwiXDBo4MyvMbNyJg languages/oxygene.min.js
sha384-lZeW3kKuvQ6QvTCaQNIRL5skdAX6vbNy0rfJcS3g6hf4pThFfond7mTVqNXsI/No languages/parser3.min.js
sha384-o5o0jdmqHdnM5OGVnNEAgoY2ER/rJBD3C9rNpw7KeNwiPWZrmq7dCndQfL//1Q25 languages/perl.min.js
sha384-7nwGyP590w+biBBXeVt4PzP2cmVUqDVlHxqdRiqLjZvyh4zHpeDlwbNmS5zICNCz languages/pf.min.js
sha384-ZoGcNrg4bsLhFfphteWbIK8npf5afw2GgtbbF0S3PEMbFdrE06u7mwFKz3zUFUBr languages/pgsql.min.js
sha384-bt2yub5fsE6mjVP+QF8CZT+tegUNlB0AT+GCaLcz/BBeshdvZ0NHyt9QJxPD0hz8 languages/php-template.min.js
sha384-aBJrhXRI4h4rGnKlFkaskM8NtwO59wanfMX8pSb8Lt/CsjGrnOZ3rRkfrV2/6tdE languages/php.min.js
sha384-cBzI0Zprf5xQZe055Ct7IMh30sL7BDUSMyHvugX+06ZW5tt1BUJ/MXqXup8HBU7w languages/plaintext.min.js
sha384-tyvyB9r09cVgTwjIKadwbgCwtNvX0yLKSwxiusoKZCKomhAhu5DB0HhlRJ1UEaR+ languages/pony.min.js
sha384-mZJzZjc9HSslCz2u8W0/5IMsOTw5Jzq1ab8VNdRm0wu7L3BR80Tql/21+QqD9hdf languages/powershell.min.js
sha384-63wBnh4ww2wttZh6fM2CDd3DzqmdqWJ9mOQ+bFMhIsUv2yJqikkdg6zw0mOSuzNf languages/processing.min.js
sha384-rWMicxHTSReGvu5AghLw4MhHNP17dSTV/PsYP/ZiZNm7su755YUIoXXASPzQ/fZy languages/profile.min.js
sha384-RW4SvUqx+WCyBevsdD9WzKnaBzkOd0xpH0B7XkyhZLDP4Vh4Kv5Icf7v17hmNTVi languages/prolog.min.js
sha384-XGYWENFCL2mzUb1XvrMayA24Zis+vjKrYBCC2hUS/Ybrm0p9eqpmJNCxOFiGIhM+ languages/properties.min.js
sha384-Qx5Ro0q2ppaDBMXWPKgBbBagmbOStvlL3+a5YxXyunjybyNtOnZkAkKzzEhLbUOI languages/protobuf.min.js
sha384-a9EyiKE/G04tgPRyrmK3XYdvQ6ZsfybD2p19+ePNoDFXKwm6uDNEh7ctJNDCc/9G languages/puppet.min.js
sha384-DbFbHUngFc8EqAStz47tquBYbW/IGHq8Q7dforM0LeXvvjNC3dghv/lKadlbhUxZ languages/purebasic.min.js
sha384-1ENHj5DjhM2jjFp8tRrx5Hzb4QWe+5kALKIIT72ctBZCEln3olAaejG1nV8+ffk9 languages/python-repl.min.js
sha384-GKucLamJwU4T0kOD7hk4VmST05mIOaAzkGjZj4wdR4m/UEaI6HsBnuk7aiK/wfJ0 languages/python.min.js
sha384-H4l/KNldQkTegZqV917EVU2Xuu+bwsm6qWqOlohLFUfA4fMcQl8yDaxdWQxm8c1v languages/q.min.js
sha384-E78EiaX7QCPks6IMoVvH3vLkuUoNVzve0JY90t2Gr7mYopBujqAxuQpUFMSl2vps languages/qml.min.js
sha384-/b1sLg6nUXVy9aLWm0NX3fP3nJHNcvdsGdvyfEY94IiEJPyZRuzuBfSZ7VR3rLvF languages/reasonml.min.js
sha384-hw6XK4ZS+hbK7xM1pJLE1nkDRi2SeZ5pUCV8Lzkq5AXeJuAhOUutburkW3OHYeRo languages/r.min.js
sha384-HkitSICiZjSAd1q6Y7p7HB+CciszXEIbS7kwgwUlbWeRdzbDwZEiQ4M1y8MX2UhL languages/rib.min.js
sha384-XvCGedDzV0v3fGIxbI6vFFqkcSkh1/lJqS/pBAtoUfI6y8jzr34r0ej2zn9qrp60 languages/routeros.min.js
sha384-+h9sgIi0OLjoXUFptjftdXSiHC4QERwf4EAVZiN7mk4u5kQL2EYO2j/wBDoD/WmR languages/roboconf.min.js
sha384-e8v/w7gpN7mu+o/5ney+CZKKMWJAG8Al2WkU30F68f1+1WpJBcyU3NCWWZ//oy1D languages/rsl.min.js
sha384-mPvQCMAwHqRewo9TgnXRbF/H3QKOwNu/UZuiyWuyS8fq3RcaMWw5J/oifaPZViKh languages/ruby.min.js
sha384-gRPoJmijUFkACzaoWKlLDNb5S8F360bbp6Sbbkzvs05Pw01QPbNLC5MHH8VukL4+ languages/ruleslanguage.min.js
sha384-/yLz99rpq/KwHMHk5LzlCodPfNn9rUBNQTxuqH4Y+FQQLGqhmF9aF79mY1J2GJim languages/rust.min.js
sha384-MU8RoWmNTxbIQz/hJPr5Oy18k2vGQfGXqHimaI/UsaPX17/V1wMH7joWxGCSsSDc languages/sas.min.js
sha384-t3iTicaNjPlFntdjQrYbTFOIG5XD1b82ae6HwIyE5XRkZEHXfWEqrIr0Mfu6iKfb languages/scala.min.js
sha384-94zAVRgXbOeKdbVaT0Tjj2J5HyL4l7dHEsSlpExM6sdHdXnXWWMqzqo8eclypvpw languages/scheme.min.js
sha384-6JVdbaLJrKXwADJYu1DnXNNXtGAj8boEAVAfODLKGZ2xAwzHAKF6xJP2+VYmtAqp languages/scilab.min.js
sha384-3dO4Ngb09QBwiTybt9YS7+LlkJC7qIp4wh4xBprmK9nGVFnE9JHqf4x41RdXo0I1 languages/shell.min.js
sha384-FhAEcwmzjrS/MKidArjKynVOgeEwg2SbwjvdBYnGAyiJ/XPOw8kfgdjsLPEWnPc0 languages/smalltalk.min.js
sha384-rnstmim1xd7Ab6djSq4Ityy7Vg2r26C8MNlnwbIPv6pS/rOSWtYOJXSFwch1zKNq languages/smali.min.js
sha384-0RiDhyevtMWTXPdwLRt1n9dnUlqEew5vXD0ucEpqYFAA7qfP//y5JhDfwP7KBTOQ languages/sml.min.js
sha384-D6wnHUaHBTRZO87FTWthJSa7+/WKg0Px5w4hdpkLi2LPuuE4yvGwxqivGFyB4yI2 languages/sqf.min.js
sha384-1vngqkuyHmkeywEWyKSFn3LwGOI51s5mczlZSjmSggdjPbrUhbZb450FfkMBRmwa languages/sql.min.js
sha384-HCD3JL30eeWciNLhVpOxyzTk6NZ+c8CyQZBGvWextTfxBpshREXaZb/WzuSMSuE2 languages/stan.min.js
sha384-P2y+SrQEOol8S/iFje6upGzIpSOhHohHWQQaqtVVtGBGqQv5+0oKEihjkRCHlEMe languages/stata.min.js
sha384-irX7Jas1x4PEtU5xw2exT1+AjGpMsf2CnrxvGIPJ3btlX0b89fRDowRNhkJdFUHy languages/step21.min.js
sha384-9reu6l/9AWOuOdpuCgxXMP1ST6tIjPZGH99sTz744VKWDe9dFtZ3d9SE/1rFAgFB languages/subunit.min.js
sha384-aw5815XHzVNWSOLi+hRr05E8oWWihqLMxh4Or4A+vJmRyGTwTHfMzxlCGMT7vM/l languages/taggerscript.min.js
sha384-u/hat9Uqp3WmjZ+l7mp06zJmJhTzvb2FRSs1LR3xz/8mJOKBTwxZ1FxI6pGLopjR languages/tcl.min.js
sha384-57uzSQzGBnsOjmrzFlmfWBO4B6b8fOMfoks5zjs0nNQ70yJWZNfcdZ+e2h5tjEMQ languages/tap.min.js
sha384-iL9CLdU0bsxFkJTga1LF/peP22S0eIssJI6y9fDfcgb99QO46jUTKDo5QIAIMI46 languages/thrift.min.js
sha384-Dz6jYg+zhXLkWzz2aDEccQQQbLSMyi/3NhRZNSX293dFjFyUWnAJTcFRwscXR4rs languages/tp.min.js
sha384-XnKyVMRRtcOLrI24NvxY0rJLmGNq4WptkkGjKo/QVhDvFxUnhIA98d4iKKrgIU65 languages/twig.min.js
sha384-JCK1FXLwy1bgv2q2inQla16PRwRwADSboSac2Y+DvrCxwHgaufFl2+p8GlSJJ5ri languages/vala.min.js
sha384-TKCPAVRn1p3IJE86WvglvvvGl1GH9qsUaJccdjmjc8wqe68CxnAP4BBK1UbS3zv+ languages/vbnet.min.js
sha384-4PGTYZX0neyxizyQB7xvxtx3CvlKCzSGVyLQWRADLfyvyK4xk9v2UtMb5sDUnzaA languages/vbscript-html.min.js
sha384-sQ1iiaNkqQjDOWIj0pVQo5W+ygy2IgyYvaz3vspbbe1Ty35c/m5ouoxqZjKUc+Ff languages/vbscript.min.js
sha384-4wqdS5OORspTLus+8fyt8y99rHmGpIwag8yhnPf0CV8Xh5oFvIe5EVcnIjMxeZvv languages/verilog.min.js
sha384-d04xW1E/Mhba1Jc6Bm5mFpI0spFxz98l9ztY0qkBsz0OdziGSPVuKWYCy5jwxC2y languages/vhdl.min.js
sha384-R1//+cN+3SuW4l+oI8N4W9oeh2Ves8Tgw8jK5ezINne0wLCyrc/3Z+dIgjjwZBvN languages/vim.min.js
sha384-nTsQEXNTRAbGUwM9qrdcp0ky9OVNxq/GQdd7LLVYmdOQJgDECV+vsmj0KozL2CDG languages/wasm.min.js
sha384-/XncB4ZIQGLP1NHWRjx13t9UtlMxAAfbkjZM5dbCmskfQNMjY1amelMMyjdsLB9g languages/wren.min.js
sha384-H/vfjNJlL6lYqrJwSdjGvDIXeuhIwrvLDSEUNb5tTYLnofQx3s8j7VFZ7Q56sR9G languages/x86asm.min.js
sha384-tfCe5KkO2JzIRjXY3l0121qfkDAIbcvgZi+KSedDAZGKzt4epDDAYUy10HIHArFM languages/xl.min.js
sha384-yJr7yzAUIVBj+/Okx0U4C/jbkmUhyGatEacfLFuCIe11dDCYeL/el+Yte5oJyuz6 languages/xml.min.js
sha384-Ofmk6c40VU/DF+xHSQwWUVqdOPEeJBPG7v7gCftJpKDpNfNPkRCR9bSTQ71dPR38 languages/xquery.min.js
sha384-79h+t+Q8r3Fl040h6w+vRsxqNvVda/5xXx4jVQ6S0DVu/3EuhXa+/fsK5LNPeIld languages/yaml.min.js
sha384-/oOeCcP5b36w4E9XPYBN9koopEAzB8jv9AGskhsllXbJM1Z0/nSqIjh7wuUho34t languages/zephir.min.js
sha384-Fouff8pOHi+TilL8nOAuuYRprk4mIQsIEPs35m0cLOafW/CvQKKg4G+WC3EI5hLn es/languages/abnf.min.js
sha384-9qpqQoOWMm+LkyyMreoXIeu15VVI7DsgUBLAjEri85DgKrWCiNJkD2o4N1wCVyYC es/languages/accesslog.min.js
sha384-uSRujAoBXlzf2MtSIBkJYGLOLsDTHJjW5XTygG+HD86/xoCZPBp94JDkDXK/5DiR es/languages/ada.min.js
sha384-weDjpd38rTCbBjCw028EAOhENKqZfjIKE2APHppQxBr8V2AzaPkEI1+oewV0Rg7A es/languages/actionscript.min.js
sha384-euocoBo8Z5HKswYyGVVp2/KLJkulAqwkH0MMtUN6FqufpvokNTJNE7xdegMntTNy es/languages/angelscript.min.js
sha384-FX+J+4lgrp3hZaxDhyFWJgr1PTISQtTRUj1i388ua/XARcKWWFAn8CTdpvVHuXdy es/languages/apache.min.js
sha384-D7+47LZFhAb8n0tDSxKlP3EDBMCqe0fVJaRMrAG4yf1Wjw6XuKBwVJVV7O+/gZHJ es/languages/applescript.min.js
sha384-igh3opPTOoncIPsdZKjiQ4+xpwYOflB46IhlqifUHSZuGJhJKEeWNCmNZvJOY/5s es/languages/arcade.min.js
sha384-cGyk4by35nmo1WcjKBDuPbmFpE/z1tN+DmyEdaA0X34ZLd32SUNvK9slW1b4DtjE es/languages/armasm.min.js
sha384-eAkPNAZumPT+Fr5PHbNB/F+4JA5gRGFW21Gr1amRlVXqz84sEeG2oJgx+7pcxWRG es/languages/asciidoc.min.js
sha384-vSeLyyIw9u9yggTYNl0Dt5OvZYvYsb4lOGKJ1s/Df+WlBIz+ku8G7ULivCaIvFeg es/languages/aspectj.min.js
sha384-q7fDfIVnohTVj/W3YzLE58Lt/Xbl1HlfgR7jC0EEvzgxEV3mkBPGA7Efg1+uWLfD es/languages/autohotkey.min.js
sha384-r1O3Si3T5pV9JW4Bxjo8CA49Ky7qb9hQpkzj5M+t0s3ISsz1g1OMtx0H4+Mxq4Oe es/languages/autoit.min.js
sha384-PtPIQmGK1qrBj1eWIxiKc6M83Gjdu+6SRzQ7dv6N86x3Lb1COX2rrttFkzTUICfm es/languages/avrasm.min.js
sha384-4FD0L93uH1uYLbLpPhoj8Knlu40BWU+ZK/OI7D6i3zqDnKBRGenoo2GZAKm61Sff es/languages/awk.min.js
sha384-DepS+vp7cPngvL0TteNv8t3lX4quTbgW5CFBwwfVsQ+4zMACONK4WuFQpIV5iXkn es/languages/axapta.min.js
sha384-MiqNKF2Z16jCOkfjOwFSGnGKQkXLRb1sztPIkgTji1wun3iyTrHOfwJ1b5YomZST es/languages/bash.min.js
sha384-fLxc4SK0ojSGMAYZxdwLcyVwjjJXs1iI2Ctuju8pFdfXYUl+m58Q9YZzg9gPz+Ls es/languages/basic.min.js
sha384-G5FkeYLA0dS7m/A1fwLXVLiB8Amv6TkrvpA76qcuD3OxKVK+AEleAhhYNXZr5old es/languages/bnf.min.js
sha384-Lnwew/vtS6KZWFb45jCDB7zLT3ceYsAi4ZXSH7+OBEe18eZWPB8IMTffwFPJXnm/ es/languages/brainfuck.min.js
sha384-2tQK/lZ/2KBF4i4gAa2HOYuRXg2DB64bHkpoZsPgGZ4pPnXagmeov5aEGJgTkgrj es/languages/c.min.js
sha384-ZGtbQXauCvAtiNEghhFTUYsnyZ9vrEbmgAeRryl/WtPgi+fg0WTvDJLr7Iciua8A es/languages/cal.min.js
sha384-Ya4T7J4kL5Kaj99i+RMvzeSZTRNokGGrfJi5Y7yKjifPrN6RN1Znn5C2CZE+zlG1 es/languages/capnproto.min.js
sha384-EYyLGpGZ8KyKeuo8jgqUul98yQV2rxfTqzKJlhUgVNgUAVSDbl06X50G5Rq8KWmW es/languages/ceylon.min.js
sha384-GjwamHjTW8KeEJ4gYeEzPaZgtXnG+X+C6TrFb2qOW92t0onWjd1OJVUM6UxqlJCX es/languages/clean.min.js
sha384-w0OmFROIJcog5SEkpCjdFjcl+NOdnjZrVsn90AtY5WXJ2D7oQTsU1Aoy1OQSDT5Q es/languages/clojure-repl.min.js
sha384-HL1Kq2fBELjY5REAqHm9Q2S0iR8pqjciXsWp3KUvNwr0zzmCc1c2Dc8FTUBSbqwc es/languages/clojure.min.js
sha384-8NcuBmsrdjpIY+TlD7Mwcxd7RfNLO7tBpIOP4p8GgidGq2k0qIg4ylgKSDfqdbMm es/languages/cmake.min.js
sha384-fHt6cp9Tv8g+kGSQGzAlxTZS8EJaZhtaC1PzepqsPPNBJ+tI2KJXII3OvWStDuUU es/languages/coq.min.js
sha384-JB+mvyNNsuTfdyEbWgJb2bdF8acu1w8z83f1U9o27GHD2zhNaUPd3noQa1YW9qHN es/languages/cos.min.js
sha384-I/iuGdcfG3h+WShnOH2zuKRM3rwseyvrZSK4fg99oUvyIL2R4SFFXJFmVDptLLWl es/languages/cpp.min.js
sha384-T7w303o672UkBmeIHrps5TA8uXALFl3Kp/QxiWjQD12JKycLIGHNX6HfgcBppdnX es/languages/crmsh.min.js
sha384-FKm0/glq+48t2Xfh7c4gGQaRqZFqUDnzt2A+Bg5JgDS1ps/itqbPWcZSZmrHSN2p es/languages/crystal.min.js
sha384-R5L30oC9+vdHTddzVSLypmTPSMGKdzNPJUsRLTygFNCIJtKM8lLbSWGSMK/+BzYL es/languages/csharp.min.js
sha384-oPLVmqFUblkGSpWATheiRoJ9sUiDCcfFNLzx1jUMzDDqx9JppxjCV7pYojBl3Dw9 es/languages/csp.min.js
sha384-bidd/oLQZegrTGj0e5I0r/K/dDmjzhT/OUrKH2KVyBmHYnIFJdtHDsAy+e8WrPOs es/languages/d.min.js
sha384-gvJqyg9uIeEQFN1IEO7bmyoxZVdH2eUGRyZ27j0NR+K6m9bpIjsdcS97pIIqbyAa es/languages/dart.min.js
sha384-1V5EcTHZiJmc4vppY/U9KDFCT5u6Q7jlpeWO3l/s/MGeA4UpvfCGY1qZT3RDkyIw es/languages/delphi.min.js
sha384-696Jdf4WS0NE0Ki9x0nbdQ5pCgPZ899TYlsQIsxxR3+FyaeqWWKKv9MCC+DpTQiH es/languages/diff.min.js
sha384-kBO2Txpy6r50WkI0azRqbh9DQfJux/pdW1i195NDnUz+G05X4vxb/ek/g9h4Dmx+ es/languages/django.min.js
sha384-TQiniGITfSp0vwDmQqgUyPqqpTq5Bn5oGZ4DH+IaZ+B9CKmFmBaFl3wDtiPlfTnA es/languages/dns.min.js
sha384-Ifvc+1Ym/F3T+hoT7FDZAHYl5t+mbwsk5Rxt+DbtFbuiHIYKyy2R2Y9FAqX90ZG7 es/languages/dockerfile.min.js
sha384-bamQ+r2vHVEgzRGCB35dUCoafgDpTiK19UPF4OG0tGn+Iop+ihQEaSxe2Ts3Rfwt es/languages/dos.min.js
sha384-xyy56LutPezp7gMky41XgA1apMwG0C3fbukwJ2U3PE3OuhyF1LGSt5/WAiM4jKeb es/languages/dsconfig.min.js
sha384-5cShFXpPDhBPfm6Sz6QwsQsRRsatV7bdaKhjpnRj9GwELRTVgrBc51EOc1JKaThF es/languages/dts.min.js
sha384-2JSFfy7BzzEtlDUPj2NJgWNW6gLMxMFDWSnR/7HiOGTCLR3IOA020AB3JOA+S76I es/languages/dust.min.js
sha384-4P/neVj4BFBNOGZCiF3NG+8TRqi1qGDJJ+jpQo3qpW8yBNuvu7zh+kC0R5Vqa9++ es/languages/ebnf.min.js
sha384-MbUs47xG/ZVpGUd9QAIvGd1H+uWoAZXEjJNFfI2KjHTseRox1KSC6ebdFudeQ74s es/languages/elixir.min.js
sha384-0uNFUI9ZwNcPunZ0wMZdB2FkkKdoq+GZUwfKOIaPnN10R/MIvTmjfbtVSOI2HlJw es/languages/elm.min.js
sha384-vVs9/XXQtY+QgND8NWSHtWZuDw28yclEYLz073c+peUtnV4YauNzgbfi5oPeLg8p es/languages/erb.min.js
sha384-e4lMmfMKckZ7jbsFoZqypFUZWB4teE6BHVa/jwz2laepAvp5QLeFib2xj99I/tgV es/languages/erlang-repl.min.js
sha384-ioXYu+++pQY/PifTobu2aLHWpyeS2pxQ5dxNgJTJlLOY+/xHR4n4rsyvcaIaElO4 es/languages/erlang.min.js
sha384-5OGcgLuzc6N326ADfI5fL80Oa9M6wgLEasegULV6HyNLlw+VR75ebeplD7bZTOK3 es/languages/excel.min.js
sha384-FDs06sb/uT5FkuVnjoUm/VA9FkgujXl1uDAQ/zotBhCBXkmA3DDcri7uQpdJohk2 es/languages/fix.min.js
sha384-TlxvypY2ndUT696gktwq0lMh9dBhsO3JzbKgRCrBGVUlBD4lw7ZJBFyL1szkj0Y2 es/languages/flix.min.js
sha384-xaaU/p5F6bYqzaPgX2rE7eJzU9swk4Htp+u9++bNshw+5MRDoblIEIwcRhaiMkIY es/languages/fortran.min.js
sha384-ut5BRRDqo8HL8GliDyVodm17j/HD8qk1dTmdpHQY2e9txutZLJd1p9ZB04gt7/XO es/languages/gams.min.js
sha384-YDJ4p6hCyKdPWbj/fvYS8koLoMHjlQ5d+DDp1EvZlJmPyysPICHcd21yJuHVNUZ9 es/languages/gauss.min.js
sha384-62u7QvvdRoxGrqtcJSE/7ereeqYd1bKEjbL5NvLx2FZUhDY+ENYBa13NmyQ9YbKZ es/languages/gherkin.min.js
sha384-QdQAPVa+4V5K5B0FxrOsd88yWTnzq2RxF5N7tkBIC2rtBupp39MhhDb9G4AFaIoy es/languages/glsl.min.js
sha384-hO2dJfqzIFHB5s2pwApsf3wj/DS8yFbQuCeAtAhzrPZWULuZE8/KN0PpBoN666Ii es/languages/gcode.min.js
sha384-+rjcUHkIwWfMFy3AV7hfC4drStiPZ12bmzIqw33Av2XwQfidsnEGKmJhbtyEPzR/ es/languages/go.min.js
sha384-w3+oAuNNSqjOWIWZ5WR7SxevaXBBZZ65zSKeTXsTz9xDf1VGoWOBp5mr+WNlQtHI es/languages/gradle.min.js
sha384-RPQQkUFYd9bQfc90In63UG41VxWgJvK4l2vqST3ZE2hYCy0209HIv2RQJ9YKuEta es/languages/golo.min.js
sha384-NjySu9ciqOORipqXHNej8QG2jAjjbr2uR4Q2Svo4vokkm2cQbW7ey2H/EmjR5bWo es/languages/haml.min.js
sha384-oKCJdP75nGYzJz3GMSBTnWfbJXuKKEJVw4S2MAvaRk/SIq9zmjxqeAl0/wt3a4KY es/languages/groovy.min.js
sha384-hDy2eeZzPHmulMWP+fdBuMrw8gGhp1pQhLtlJfbt28d2CiKlO/1kiD5rNFyMV3eC es/languages/handlebars.min.js
sha384-58MKJOjY8pPslZPUgzPEmU6J44BunA6RtQyJGjXObajcTfVxS2XdxB5lmjcuSwIr es/languages/haskell.min.js
sha384-YnHmHQ+UDSdT2eqAvWLruqaB/dbNihAvWoLrbUulFAmJE6E13NPsP57yPN8INsJ6 es/languages/haxe.min.js
sha384-7pNV1cMaIM+LLsXGihSk278A1hBLT4WmJwtdVCt2KNOm95ApQLPxD3Seh6Y8TTg1 es/languages/hsp.min.js
sha384-FUkricHUr/gKzzI2ih3XaHpKEeXL7Zid2J+10fybtnt/uytkyseLkEdDjJGejj+k es/languages/http.min.js
sha384-XDOLfXWKmQhBks5+k9MYITZRt2JqSzQwuNfF+SOhkdkQ1Lnde9dIcWEFP+wY3lta es/languages/hy.min.js
sha384-37D1hrtos4Yk4jTN3DQggvgeXlguPXcKuUX//smBQLzW34d9j0WbJFQjYPjyN139 es/languages/inform7.min.js
sha384-HC357HQu4DjeTnFt5IsWD9XmnS8+Rmrm5UEC3ci2+Winm1pI20ffSAGar8VI8BAc es/languages/ini.min.js
sha384-EpKDRxyDQ3UC7CVTp3ws+BZjdUk6MQ1WXjRGVh9H0FyFdCC0FZzi8dSJhZPSrZUp es/languages/irpf90.min.js
sha384-6JKESiwCKZOByrYt/lfygm1KDGRV4BcBLYPo8zhtGKt9VmOITxJaD9WmIgfctSgF es/languages/jboss-cli.min.js
sha384-lz/oHO+ZIEI33yXxefAbXK0j0kKpmH/62NQettgmIGPGj5jcyjEmGzAZYSq2frH9 es/languages/json.min.js
sha384-jxO6+7sDQD4OfVdkf3BHrkDhzfAC68k0HlRCu5pCX7pqUpmoPdtvVSHeNuOoAiKB es/languages/julia-repl.min.js
sha384-VkH7vgcyynGHSlfgkE/ygvo/6LE4+pcxyu5fNwCnRQqZEn6w62UM+3sVZMKf9/oN es/languages/julia.min.js
sha384-6KGNfV/gJSpT7Vir3mKvBFaTcQgHB5T9IffSGpQ/Uzjsfp0uj9oev0t1rPGeqrMa es/languages/lasso.min.js
sha384-KoNHiiCDV6D1ZuI/0h1PSZ3OKHGI68kz7VWcYnn9BBe5H9foMzMUuAMPRAvYemxm es/languages/latex.min.js
sha384-y6WSMAxRSpA0tIWZJIy2V+cqF/Xzl6NxNn5ZMrKN4k0BqSRi4TdmdOwhHD60buIB es/languages/ldif.min.js
sha384-5x3Cy6/NcphBBDJcX7NJoAlmpcT1qUUjooVhekDF7AQj8WlZWoJODlkjkf3AU0UZ es/languages/leaf.min.js
sha384-2wx0Ck1EjY9ad8Xz9XtZdS4RZjZ2QoMDWAsByW0YGBMRvWrA/8KcMyoheG3PfDUE es/languages/lisp.min.js
sha384-0TiB12m9aPmF0Z7WAACL6RL2jHeBOluDehhRhHa05925+aXIsQsVDoT5KOnbqjzJ es/languages/livecodeserver.min.js
sha384-HIgvIeUa5b/1hLq4OBM7oZDpy7ZkAa28N3tyjZD7eJ3mJGRASJtoQ3ypsXtCzUfg es/languages/llvm.min.js
sha384-TQ67y7Kf3ShVeBlsafghpuaJkcG/Ve4i+aEVV4aelYvm6khG/KLMTkJC1zTj7M7e es/languages/lsl.min.js
sha384-7jZdXvYYZ2+tElT1bVqNX5wF8wAlu+Qn3oyz6DcQGpTPgBFm4IU+3z2bwZUic8Ue es/languages/lua.min.js
sha384-46tuvnKzHcD7ieHBaYzwtWXpWjrifvO7DSfB4vKDbHeHqmiWp3z4eLhbAeZzIwuy es/languages/makefile.min.js
sha384-xe3XeCwsjnLrzDFNeMpO5a/5cZMUQ+cZ8VKY4y1OPq8uw6XPyGyvusp2Vi9T42n4 es/languages/markdown.min.js
sha384-w003im6ErnZggvUuqXKzrNHG/i4fK0PGKmQoFjAaqOzR11KVscZMboPn02qKCBL4 es/languages/matlab.min.js
sha384-D/xG4PFmNvqjO0bO5DVLte0/UeF5B9/CZbdeZxpbGPK88U3Nk2Y7K+NLR6Nec9rV es/languages/mercury.min.js
sha384-EaITFhBPIGxf2CUS7H5OAcT3YqTvRK7XtyQ98T77v4bw0lelGR0Iubm33klOUyAk es/languages/mipsasm.min.js
sha384-rEq47tbR/rcsIXwmgXglSwyHEQOZ4mPo5NC/rPy7d8AnpjpPVZCrcio6mKjpyqxM es/languages/mizar.min.js
sha384-ENzkX20Zass3fkTcr2hlvH6533AzvJoTvGa6IqAwox4j2jkhT2E0SYl0ZgyHAAAc es/languages/monkey.min.js
sha384-HdWkAV2WWYhP0jZhtGqaQFHPGkh8LS3NbAgbTNMF0mToAUqMS70JaBjhCi10+3rw es/languages/mojolicious.min.js
sha384-r31Y/phDRowwbvZFv8VmUoFE1B9mBKW/pyQsVsxWEArgemITbguBuamE7ykdraox es/languages/moonscript.min.js
sha384-0GFqYGVC9I4SWvWqduQNiGmdKdLxw44UwbgE3Y9YS90R5a1v60BhYFaNiBAmBL4e es/languages/n1ql.min.js
sha384-yJYNUI7ee1kgyr0zxZPh9ZRk4gwVtuLWX1o+2dBLjpNzDrcA5gHC2u+l46ifLMVL es/languages/nestedtext.min.js
sha384-l98mHgZTgmECjAMaCL0iJbnVlKDCmkYKp88MZopZeZAVf+ugxiomIV1yxnWJ2Rnp es/languages/nginx.min.js
sha384-9CySrYgMLCP3nJJxkJxuiOPpXDOZB7Eo3oRPZPZiaiewDtPJrx20dmMgMCVuT4On es/languages/nim.min.js
sha384-rn2Bl9/7vEfUxkM7HPykWVA2c8iltCQqAhYzUuZpP8EuZFH3wqkHm0ji+LfcBTtE es/languages/nix.min.js
sha384-q24p4lKsBhrzYpRj5wY6tfKyZ6kwulsV418MCOt1GLK2QmjPxnPxYkZpF30F2sIo es/languages/node-repl.min.js
sha384-xYQfUALCiuxQTPKf3y0hXz4biqWQ2e/Dg110YmjK1ZdHV6/WQNvfnaVK2anTGpTr es/languages/objectivec.min.js
sha384-Kqc+RuKIr2aN7O91EgGtiMyX9noM0+o28VWtCdrOE/xjslr5pPiGIuYdmUMUqnrp es/languages/ocaml.min.js
sha384-AG8H3IxfdCVu/KpRx9sY5Vm2Fx7gmGEqkxXyYG9HliRSL8byoXdGZE9IU6HeCVOJ es/languages/openscad.min.js
sha384-RpV4hCVFcvYOpOG2z/uf24Fb6GFzipjFwZa62yb2741uxLGjfecoeFy8m3l7/SdJ es/languages/oxygene.min.js
sha384-/s6Hgk1kqu0BCRQC5e6FhQ6YT491bQkaHWAHFUSVUgUYhrfxxSCN8ydOvR/c/INf es/languages/parser3.min.js
sha384-G7F0wkZLr/2v2BiPiXS6Mw/WAlHoy5xjl8yP+bjHH8z4u2ioluPMZGeDO+wPkWqS es/languages/perl.min.js
sha384-EhRrVYHkix5Sdiyjgiut4vV9ild+sNY9qFJ+/QTDmgh2PGd1P3asfA24g9TYbLK1 es/languages/pf.min.js
sha384-ggVRolCeFKJ8FCELj+sUeJPwUsg8wsBzwisMnBeTOubOA1nPMmjjuBMIqWgzC6yH es/languages/php-template.min.js
sha384-iXx9L3sMqAFLUOlXipqGBP4crcEuoHw0GLKI/NjHwK5hNu9Lq2J4YSru9e3V3Utp es/languages/php.min.js
sha384-RO4/WpU+eaSwSSLNfwa+PapSO2V+AKYvF7hsiZUyPdeId7WkoyqSYyiqQ7nyy+27 es/languages/plaintext.min.js
sha384-rIMfTHsTvx5/7JM6z1Y+PEtRsRzgWG/ybOI2tjwT+Jlt5peHNhdfZCtpuPpUPb1H es/languages/pony.min.js
sha384-4myst3Uc6lO79W7Iuy8Jvf3KL3wi6yvgbU4GdbZrnXHoiD7zY6c4uNxIyaBzJgfP es/languages/powershell.min.js
sha384-ecJzicWucp32ZC4yfOEuOxio944CSW+aJxpOA6oRVlEDQ35Uc4K+I51A893Ll+Ge es/languages/processing.min.js
sha384-BneYJK0bN85OZlSpsKykVx+FTw/xeNItQJ6G9F+QgR31qmMGPh1TUB0ablvF/c8R es/languages/profile.min.js
sha384-sObIEtdCIZ3u3dq15SxHmLrKl52Nz5H/HIYMde7zqQx2LXscXBJsI/jF2k4uMtXM es/languages/prolog.min.js
sha384-vvDtWOhf2ORPSJHhFsXnELuX/jDOMyO+frdVKBukrtgHT0tD3FbXu+NY2OeHyP2c es/languages/properties.min.js
sha384-rqeTvvo9nZQxMgDkrUg5RQ9C0WNTNYg53riVYoAmgJ1CZG508B72gQI6h/T/mgs5 es/languages/protobuf.min.js
sha384-1nPKAdSBnfWcywnO0pZc31HDzGRLdIAXBoh31SLax63MO8Ffk76UQFIf2jhjjm/s es/languages/puppet.min.js
sha384-MY5HgzHLLbtIyVScS8YKeCC03c6uAUWizSHYFTilsH7eF+7Mw8N4A4gm91Y1lW0R es/languages/purebasic.min.js
sha384-6p4wLM3Ifz+29rFDH6n8k4pgTia1pOC9ENghYsnj3rHs9ca1SnUvhTxfJKK7mEtT es/languages/python-repl.min.js
sha384-rBWpx68pRKh/0Pe0yc4LkVGvKL+1jpcvmpCAHY4BD8MMEOsYmYluNBXUXWzHF/tg es/languages/python.min.js
sha384-GxvdHfOjsSHBFL7yRtia/hLC01MzQW7NccYEuOTd8zXKkk/jaElrQfAtJjP/MB+l es/languages/q.min.js
sha384-pUJzb7Sb8Kjyn9WYCa3jvTBN9zWvJNjfirO1UbFLSdRHx4dxlkWyf3tQdszuZPm6 es/languages/qml.min.js
sha384-gHc4QgvscIBRpvfakvwfqw8IhUcwFkajbtvxH8JiwX/LDklxU52wtPW9CbawCvCC es/languages/reasonml.min.js
sha384-4n2sRn9rodqLGSOEtJBPVJD+wMJtm8zf9XMkehsQmfXGK0jM20Wkomvj7HTk/7uv es/languages/r.min.js
sha384-bUsLxr+BMuKo6d0yK5VDUqJBw+1MIkgDCz2FlKmGLHlrDKWw4ymznEhc+jZMGpeb es/languages/rib.min.js
sha384-V0WL7/4K3qIejv4Byae/IHLEHS7XmbHC7hqdVF9Sfhu49c7dtNLYBh7/X1U6dGDY es/languages/routeros.min.js
sha384-+COso1r2cnYWY+FDXHcl4ClWbZixkIXKkbkee7pZYbPffirWaJhZgZBSQfw+iqmB es/languages/roboconf.min.js
sha384-QBJ1DidlQRW38ZuoH95Q6Oo+0IQCi/g7Iqqbp/ab7JORZ5ZchEFD27iL2v3sNZ9x es/languages/rsl.min.js
sha384-yNJd8pZrCYAgL0zTVcJe7zFTtxllXc5CwCS3kc99ppPMtzINjZ8JQjA6IhlJwz7W es/languages/ruby.min.js
sha384-uKSZpfglfcwoinP13jNAy31DS3cVIF2UK/UWYuFExielUd/NyjnZpHy4YzaeYMl9 es/languages/ruleslanguage.min.js
sha384-/elKAwXVGeM/fl2QNyOKsUg16Eh2FZUl0Sh+6wYBF0fnmbIXMSFopI5IL2Tiu4ku es/languages/rust.min.js
sha384-FSGldU04CnPnWMIa5kZneXZcyxBMDW9yCnoBTIDeBVbyxaIgoW+j8/1De7aNkHNv es/languages/sas.min.js
sha384-D/DxJW/CEcBQEqFtuk2+w4L4i7706R1RZDDpgwBTcs/7hogc4Eg/7cYzWg5kKxjl es/languages/scala.min.js
sha384-jiBF3AwPq8kgSkwNKNqaJjRRiXeIgFyMEOvjxoD4kUf7zsdHB7akR9bHAApYxG7h es/languages/scheme.min.js
sha384-N1IVMQwVRpGZa6IpD0t2MMC7lngLMg1NTF+rPelYUTKvClIOIRMbRTx2cm0Y3t/5 es/languages/scilab.min.js
sha384-yHwrMr4NOfRTuPiPbViWPFvv4558WEO0v9tZPB/iii6w0cuO29NyiIgBLEtQlqdR es/languages/shell.min.js
sha384-oJTpbufDPiHf7PdfLawe/kulJKHtzSoppbDqnPJf6zqnx4Xtkyg3InB5PWD3qiFo es/languages/smalltalk.min.js
sha384-dAFwtXlLrmtz1IvlaidD4FSu2t06xHXPqBssuwKE5CG45/tx3TTWrqxYF9NC1584 es/languages/smali.min.js
sha384-UOp1fyR6lhdF0CSDCSRM3Zl6y1HiKryNCVbI8L9L1My+G3wl7N4NHNiWpQEO11Fp es/languages/sml.min.js
sha384-cNF08wjWNaVqUKX+PHgLBuEX374abhrbG1aP6mBOU7m+XG1gEIc1cq253uy43KL8 es/languages/sql.min.js
sha384-DS+ky31SbKwUrZNMSLP/6t6jQI1Wx3yPuYUST5Y8MNQAgBAVO1BVP8vOJJ1R9jrm es/languages/stan.min.js
sha384-mLzRX0dOpoVDKduZhYnpYa8R8ECVycDHi9uVL9yujhBwoS0ZbvNlBkNRjCJ8tjMC es/languages/step21.min.js
sha384-uaTNOIiq6Ae0bErkwLMP0J9TUwFfDprHfkrjdw/yBGrFxqF/erTs2F9hmnbTwmos es/languages/subunit.min.js
sha384-4Nzhzp+c16LA5nZq0mv552eXdpXz23ZE+f63I5JRAlB9EQx2r0f0wFa3/7hq4ics es/languages/taggerscript.min.js
sha384-faGenrL3H6svo9R0UHF1L5FHcEY84WBC7sONRDVj6jAqnTcLAYEhTu90ES4MT3FT es/languages/tcl.min.js
sha384-vZPMthoxmgYDGl+q9W4TDaLLz+JnMLsdDndZlZUaSxyxjng22cT37qiwDhChNTT7 es/languages/tap.min.js
sha384-ODdQjq5KfC6fZiFDeqlSQOhL3lle2Ha3ddr6HumuMf+gqG1Bfa+BX91wiC0rNEC+ es/languages/thrift.min.js
sha384-xYbjtQfqC7Ey3spg/RDQIdZjPHFHUryJ4zTEEKJpJHVIWgbQXHGGzc62Kr5V5Kce es/languages/tp.min.js
sha384-TOaql2HZyrfs2TMCbYLBKxmoJN6Jw2hBYVGYm/Ky2Gi6puOaf08zSgPGtTClDOzu es/languages/twig.min.js
sha384-zv8aPUAItPsVA1kHac8wWtEJzuy4A86JoYCX3gbf2NKaIgaf8wABUWdld43ZVews es/languages/vala.min.js
sha384-CXPX0czeVMk5MMDQvTGiUApsDXNE4x04JgLPxZiWzNQ3lPuKuQ8O0+XQcwgupq/Y es/languages/vbnet.min.js
sha384-tIXuTswTPrFBI84UbLF05zPXqHjZC8x8nlovaxiX9VCLekljnfkkpa3rCt6PgBd1 es/languages/vbscript-html.min.js
sha384-HDVus4v7aQLlKN085TfSy/lT26B2IhDRUQjl4WJx8JBPxBkmttFFPK9qsqRjfSBF es/languages/vbscript.min.js
sha384-d/VRenWoDB7qr/RUBWextKiV8u1a06IHKl7nkXmvCJtnWx1Fw4nI/+rpyx2FF4s9 es/languages/verilog.min.js
sha384-pI5IRxloAOB6rsSlIizTbg2E1PMlKj3cJtrgM5fz/t8HzkledjVwmieBAe9xBxhG es/languages/vhdl.min.js
sha384-zKgdSZHGOnEM66oNKwwwJtjOJM3T8TaEDY3vzLn5sCgCwRat2bvwkapss312Q5Jy es/languages/vim.min.js
sha384-S0qmXGSwt/WPtpToZcZykjNFsc5ZvTPxFtNSx1NCiVc/frxQySx92uJJ2HKsS3uo es/languages/wasm.min.js
sha384-se9bdWUMDeTu33ctnBlEPD13GyJbzheO7dBEUfquHpxJpgIae88S5I3EYKX8IyQB es/languages/wren.min.js
sha384-lxHETAtURCV8kQa5PrCzzmdwQQmgwIZpGRJIromWWIfQ+xFfxNYtICnc8N3is4wB es/languages/xl.min.js
sha384-4AvOzycKmAgJ/o1FbeBD0eIUf7TSRq0Fo2nVWSA3Mje5THrG6gkWkbShxgpSAvzS es/languages/xml.min.js
sha384-hDKILhoqG9K05Ip4usZB+zRBt47/Eou5v4tAYFUMg3BtUDb+LNEwzHt0c4jy8GSC es/languages/xquery.min.js
sha384-riS+o1ezAsBldQLNQ6TAUOgMHSaSQwWVx7hG8iBpBw423VO4IV2tX2pcEEmvttzC es/languages/yaml.min.js
sha384-6kqCtrfIhk3SE8aJUW3nSl4MKHSBUoTR9JHh1UKm236IB8bjCS1mflNvBvP5SCQo es/languages/zephir.min.js
sha384-kyyxx37iblbK5/kTzhqEdaVZrf2johTlsnOZKY7ga70TGfLoacowMnm4mgVA/0WN es/languages/maxima.min.js
sha384-SBtE+Y43sdQVjqQYXpTrhGGJU8Llq886+3wMfHMiVVEK7qyijcsR41vJn2EZIBq1 es/languages/mel.min.js
sha384-TquWlEbohJSGgtncjNLfjOe5MpE22A6fW2V/CHf6mHp/+OYdS9ymPQ9sish+3fUd es/languages/pgsql.min.js
sha384-bgUpcz9oSZmrLctbrZtX5mHbMIkmVpDiAr0QFalPj4aO/WDQ4MxCCpmrBnYKEF1j es/languages/stata.min.js
sha384-O/HWaCCJA9vbUBKDfN38QMwY0lkj48ow2cNoCqRxa6eQ1zXIp3sDQOmJUtjjF+Kr es/languages/x86asm.min.js
sha384-EwypCRvfyOl1d/2YMvIObI11+fO7M+mKMHwoPVd7hKYFuVfGGQTuIQsx3G8wE4Rb es/languages/sqf.min.js
sha384-9yhmKfbYEYIH4UYFANicIJvhulGUcVsbBceAfKN1ydTZl8UginrVuGb5x6rHqoht es/languages/gml.min.js
sha384-X1YbP+5+8frZhhrLaOv81nd0bFozt7xZKfYrFf6wa9PcjGAUHXSVX+3Nu+p/qm0H es/languages/isbl.min.js
sha384-r17pFnKWTrx1SstcROSWgsJ2FMoRGSUNN6pGr+LHUbqtrwvjZXB+l+E3AdDJsqNb es/languages/1c.min.js
sha384-tu8mHDrYoUezZR7t9dGCBQ8TmfKQKDzX06Q8hbYVldba1EHPleoIKPUoiQp0ORD/ languages/arduino.min.js
sha384-le1IYml843ApF2+vrbN8mA1dkQAHljOzfpARlAZZOC7aBtbZrodUqdBshNpU5uqb languages/fsharp.min.js
sha384-oay5dqT0L2hJKdxGb8t8qlcQfo5rBEj0dwIxjxSsqRRDCXfhcS7uxSYwdVGY5tGH languages/nsis.min.js
sha384-NwfbCFcRMexa3lGTrxy8+ifJMybt9jZROM6r7mHSZsVMCqCqIWaE1kN11BZ1N/ni languages/coffeescript.min.js
sha384-HvjK3VuBr6gFlU1nMBEecLHzJ0IDLgt1J8UxfLW/kE0D8vjbjpB2JIIwzjLmd41Q languages/javascript.min.js
sha384-2pR6rnBzy8oSQBDpIsAG5F0TEvm5kpLX0fe4JEJeWGI4I7oD/bNJ9CopzI2HwF1g languages/livescript.min.js
sha384-FZHmPZkFy5Symr1Uc1gcko525cSEgMa6rby0HEVdiHkZBaG7+wc9BQm2h+RtiXdu languages/css.min.js
sha384-ixzRXWGnbWBwmYCgN7a31f4nDPOI8MS/fIZ0tlkkyvRwZbhGJ6QY9Ud/S/eJPNVr languages/less.min.js
sha384-+7ejmiGzy5fsc68XdAi71pOJmHpF+Yh2049aj5e9GVSyyTL/tj0b2i96JqSmeYfN languages/scss.min.js
sha384-LqVq57LY43G8aL4u5M9C0gIWZOw9leqUcDhjb0XTJ69HbRBwyMiBnm8Br3Ci7Fsp languages/stylus.min.js
sha384-KN3X4elH62aI+CF1lUSSfHbNstygiEUGAeRUCz1kx5L3Qw8dKscci19oqEHF0bTf es/languages/arduino.min.js
sha384-2n6lD8dXUmLUUDQUpPLD4q9bcwBpEsboGfs4rYH4klh8xxUpxY0WpU24JApbTw8y es/languages/fsharp.min.js
sha384-MHwimpFiTTfNPK2psyMKSRctpjyg3Y4CbCRezm9PN5uWXj1tEtul+XEaNLWw5ylm es/languages/nsis.min.js
sha384-sO8B3dVY9aQz9Ie2Ee1r6yWbwa0j80h8wpUlIMAcWBBsGD95Wh84KYpZue9kgo/q languages/java.min.js
sha384-wyyQ9Je4RLGqzFMpwAprwnvsg0/54GFVwrQveZmwJ2k4ohBMA0r2LdM+QLCbfvPQ languages/kotlin.min.js
sha384-WETaB7jYMOmOLreQZ1idSqsnLnJ9Na20K683pZvSEjzej3lFFVWq8uEhqOJmO39a languages/mathematica.min.js
sha384-bSLeHjoICcjSbPZwWiIbeBYHswnDI8gJKOOxuYVq6LQOQ8roIcplPRpc+fLD5VOG es/languages/coffeescript.min.js
sha384-m3WJ0RHCiy3NwtPLTfIVQrFSbwuwhEHmdUOOtgz3NTkgKD9CtQagsgPgOx3V883E es/languages/javascript.min.js
sha384-zJbYV36X6oHdpF2boVwjHJqXZfUyuhbfOHY3OEHW5drpfztwIWOIgUKVjfj8l+m7 es/languages/livescript.min.js
sha384-QaUVlJ65oB59I4g0W5gcZW43+UuEDlXKIdg2InO0gFXepJY30TFJhgv0RXOaAdJn es/languages/css.min.js
sha384-nA5xx/igdaX1x3UZbnIE+n/7iKlY3KMdu6SRoffoEvLGoSQS1Qqz5knsJpWG47hm es/languages/less.min.js
sha384-a9HlunWZvy+aJwAWEaQM/Hs94JgJXAZcrcPg0eEqqaCRKOuV0RhnHq53JVnbcGi8 es/languages/scss.min.js
sha384-OArMEizrQG0gpEW5l1Bq1a4xrK9GkSoPInyGspKiZc5Kvelyve2GMQru53SZ8bEB es/languages/stylus.min.js
sha384-6MtjMU6pMAuuUUghnVxaH7itlvw+MFwK6OVzSAXi+gJFpkUH4LcVRZAfpwZMbX7M es/languages/java.min.js
sha384-o7yxu+nkVsNjW0BSu2v6flwesuDuOp6jqer9YqCSv888Ua+cbnveD0ADwoG6HVrX es/languages/kotlin.min.js
sha384-Bsd2lM/UZFBrQxxsMObvG0utdxjfKvC9z3AXYIPbxN6Jz6yUOwtiSoa2R2A6c1Kh es/languages/mathematica.min.js
sha384-+qeHk2EXabpkP12cyZy86CUsjrx3vkATJOWmSe7CimbohZEgbBS82bW1V1mhUQgS languages/swift.min.js
sha384-L9wp79hAaUL3Few3++rZWRD/1xauP4TghDQwD9vwhbd1c7kMGvySxnO32Tw/Hn2m languages/typescript.min.js
sha384-D0mLFdLdlYo5AGDmiaWF6JgS6Ko8WTNWMytWazNcaAV7HMI2MuDRjzqX1/r97xdZ es/languages/swift.min.js
sha384-foiW0uL2MSQo/94eVpOEalNqvCZsOIdK9VXYR+c+drZxjwdpI5/MFBSH7PdzwH+Z es/languages/typescript.min.js
```
